/*

 cyphropad.c 
 
 cyphropad (pronounced cyphrapad)
 Written February 20, 2009 by Marc Pelletreau

 
The algorithms presented with this code, and the visual output for witch it creates was written
   exclusively for the OGI-LUMEN OPEN SOURCE CODE SUBMISSION CONTEST (March 1, 2009) and all code released
   under a Creative Commons Attribution-Noncommercial-Share Alike 3.0 Unported License with the sole exception
   of the function "cube_root" which originates from Wikipedia and is thus probably considered common knowledge.

 
 This code is intended for use with:
   Ogi-Lumen's Nixie Duo and Nixie tube drivers and arrays.
   The Nixie Library 3.0 written by Lionel Haims and for controlling OgiLumen nixie tube drivers.
 
 cube_root function code from wikipedia using Edmond Halley's method:
 	http://en.wikipedia.org/wiki/Cube_root
 
 
   COMPILING:
   ----------
   The following defines can be modified before you compile to customize the behavior of cyphtopad:
 
   DISPLAY_LENGTH xx  -->  Set this to the number of nixies you have.
                           Note: each nixie tube requires 14 bytes of RAM to process.
                           70 nixies x 15 = 980 bytes; Arduino only has 1K of RAM! (see the notes below)
 
   DIGITS_FALL       -->  When defined, nixie elements fall, when commented out, they will rise.
 
   START_WITH_END_NUM -->  When defined, start the sketch by displaying the END_NUM. Comment out to start with random values.
 
   BOOT_WITHOUT_SELF_TEST --> When defined, Don't run the all tube self test at bootup.
 
   USE_SQUARE_ALGORITHM   --> When defined, falling elements are processed with the d=t^2/a (square) algorithm.
                              When commented out processed using d=t^3/a (cubic).
 
   dataPin xx   --> ogilumen NIXIE Driver data line or SER (digital pin of the arduino connected to the nixie driver)
   clockPin xx  --> ogilumen NIXIE Driver clock pin or SCK (digital pin of the arduino connected to the nixie driver)
   latchPin xx  --> ogilumen NIXIE Driver latch pin or RCK (digital pin of the arduino connected to the nixie driver)
 
 
   RUN TIME:
   ---------
   The following variables can be set at compile time, but they can also be changed at run time, on the fly
   through the Ardino IDE serial monitor, or any active serial terminal you may have.
   You may of course, find it useful change them through your own code...
 
   [q/w] CHANGE_BIAS   --> (0-100) each tube has this % chance of changing all values after it completes a cycle.
 
   [!/@] STARTNUM_BIAS --> (0-100) each tube has this % chance of starting a cycle with a lower digit.
 
   [1/2] START_NUM     --> highest  nixie element level a tube will start with;
                           tube starting number will be between 0 and this number.
                         
   [3/4] END_NUN       --> The last nixie element level in a cycle. Set to 10 to have a tube turn off at end of cycle.
 
   [a/s] MIN_STARTTIME --> The minimum time a number will hold before falling (0-32768)
   [A/S] MAX_STARTTIME --> The maximum time a number will hold before falling (0-32768)
 
   [z/x] MIN_ENDTIME   --> The minimum time a number will sit on END_NUN after it falls (0-65536)
   [Z/X] MAX_ENDTIME   --> The maximum time a number will sit on END_NUN after it falls (0-65536)
 
   [9/0] MIN_ACCELERATION --> Minimum acceleration value. Recommend 80 (40 for cubic algorithm) to start.
   [(/)] MAX_ACCELERATION --> Maximum acceleration value. must be > than MIN_ACCELERATION
                              Values < 23.7 (7.3 for cubic algorithm) are fast enough that elements will be skipped.
 
   [+/-] SPEED            --> this many milliseconds between every processing step.
                              For smooth playback, consider setting this value as low as possible while
                              increasing the acceleration values above.
                              Also note, when adding to this sketch, you can also decrease the speed
                              value proportionally to compensate for the processing time of your code.

  NOTES:
  ------
  This sketch uses a *lot* of RAM. On Freeduino's with only 1K RAM, limit DISPLAY_LENGTH to 42.
  Each Nixie Tube requores up about 7 Bytes (14 for a Duo Driver) each to run.


  TODO:
  -----
  -Need method to read and write all runtime variables to the EEPROM, then the startup conditions can be
   saved customized without requiring a re-compile for them to stick at startup.
  
  -The computation of d(distance) from t(time) & a(acceleration) should be moved to a function. Although
   this will increase processing overhead and use up yet more RAM, it will offer flexibility and simplify code
   modification somewhat. It potentially willallow for d to be generated from LUTs or other methods.

  CHNAGES:
  --------
  Version 1.2 (March 31, 2009)
  -Improved RAM usage, DISPLAY_LENGTH maximum is now 42 (was 20).
  -Fixed issue with the runtime menue values not displaying corectly. 

  Version 1.1 (March 1, 2009)
  -Sketch Release.

 Last modified March 31, 2009 by Marc Pelletreau
*/


#define VERSION "1.2"
#include <Nixie.h>
#include <avr/pgmspace.h> // required by p();

#define DIGITS_FALL  // when defined, nixie elements will fall, when commented out, they rise.
#define START_WITH_END_NUM //start the sketch with END_NUM. Comment out to start with random values.
//#define BOOT_WITHOUT_SELF_TEST // Comment out to run the self test at bootup.
//#define USE_SQUARE_ALGORITHM // Use t^2 (square) algorithim. When commented out use t^3 (cubic).
//#define OUTPUT_NIXIE_TO_SERIAL //for testing.

byte CHANGE_BIAS =    30;  // (0-100) each tube has this % chance of changing after completing one cycle.

byte STARTNUM_BIAS =  65;  // (0-100) each tube has this % chance of starting a cycle with a lower digit (START_NUM).
byte START_NUM  =      3;  // higest number a tube will start with; starting element level will be between 0 and this number.
byte END_NUM	=    10;  // The last nixie element level in a cycle. Set to 10 to have the tube turn off.

int MIN_STARTTIME =  235;  // The minimum time a number will hold before falling (0-32768)
int MAX_STARTTIME =  400;  // The maximum time a number will hold before falling (0-32768)

int MIN_ENDTIME =    170;  // The minimum time a number will sit on END_NUN after it falls (0-65536)
int MAX_ENDTIME =    220;  // The maximum time a number will sit on END_NUN after it falls (0-65536)

int MIN_ACCELERATION =	80;	// minimum acceleration value. Reccoment 80+ (40+ for cubic algorithim).
int MAX_ACCELERATION =	260;	// Maximum acceleration value. must be > than MIN_ACCELERATION

int SPEED = 20; // pause this many milliseconds every cycle


#define DISPLAY_LENGTH 12 //set this to the number of nixies you have. You can run 42 tubes MAXIMUM on chips with 1K RAM.

// note the digital pins of the arduino that are connected to the nixie driver
#define dataPin	        5	// data line or SER
#define clockPin	6	// clock pin or SCK
#define latchPin	7	// latch pin or RCK


// Create the Nixie object
// pass in the pin numbers in the correct order
Nixie nixie(dataPin, clockPin, latchPin);

//  The below is essential to re-order NIXIE digits from (bottom to top or top to bottom)
//  however note, that although the method is fast, this is not overflow safe.
#ifdef DIGITS_FALL
  const byte nixieElementOrder[] = {3,8,9,4,0,5,7,2,6,1,10};  // Russian IN-12A Tube element order (top to bottom)
#else
  const byte nixieElementOrder[] = {1,6,2,7,5,0,4,9,8,3,10};  // Russian IN-12A Tube element order (bottom to top)
#endif

//const byte nixieElementOrder[] = {0,1,2,3,4,5,6,7,8,9,10};  // use this order for human readable testing, see OUTPUT_NIXIE_TO_SERIAL above.


#define LED 13
#define LED_ON	digitalWrite(LED, HIGH)
#define LED_OFF	digitalWrite(LED, LOW)

#define EVER ;; //:)

// **** display data below ****
prog_uchar bootmsg1[] PROGMEM = {"--------------------\r\n"};
prog_uchar bootmsg2[] PROGMEM = {" listopad v"};
prog_uchar bootmsg3[] PROGMEM = {" by Marc Pelletreau\r\n"};
prog_uchar bootmsg4[] PROGMEM = {" 2009\r\n\r\n"};
prog_uchar bootmsg5[] PROGMEM = {"---8<----8<----8<---\r\n"};
prog_uchar CubicAlg[] PROGMEM = {"Compiled with Cubic Algorithm\r\n"};
prog_uchar BytesFree[] PROGMEM = {" bytes of RAM free\r\n"};
prog_uchar Running[] PROGMEM = {"Running...\r\n"};

prog_uchar SelfTest[] PROGMEM = {"Self test: "};
prog_uchar Nixie[] PROGMEM = {" Numeric Indicator eXperimental No.1 tubes\r\n"};  // AKA: NIXIE TUBES.

prog_uchar SpeedMenu[] PROGMEM = {"\r\n Speed [-/+] = "};
prog_uchar MinAccelMenu[] PROGMEM = {"   MinAccel [9/0] = "};
prog_uchar MaxAccelMenu[] PROGMEM = {"   MaxAccel [(/)] = "};
prog_uchar StartTimeMenu[] PROGMEM = {"  MIN/MAX StartTime [a/s]&[A/S] = "};
prog_uchar AndMenu[] PROGMEM = {" & "};
prog_uchar EndTimeMenu[] PROGMEM = {"  MIN/MAX EndTime   [z/x]&[Z/X] = "};
prog_uchar HighestStartMenu[] PROGMEM = {"  Highest start# [1/2] = "};
prog_uchar BiasMenu[] PROGMEM = {"  #Bias [!/@] = "};
prog_uchar EndMenu[] PROGMEM = {"  End# [3/4] = "};
prog_uchar ChnageBiasMenu[] PROGMEM = {"  Change Bias [q/w] = "};
  
  
    
extern int  __bss_end;
extern int  *__brkval;
int freemem(){
 int free_memory;
 if((int)__brkval == 0)
   free_memory = ((int)&free_memory) - ((int)&__bss_end);
 else
   free_memory = ((int)&free_memory) - ((int)__brkval);
 return free_memory;
} 

struct tubestmatrix //this thing can get big; when using 60+ nixies, you may start to run out of RAM on an Arduino Decima...
{
  int starttime         [DISPLAY_LENGTH];
  long time		[DISPLAY_LENGTH];
  unsigned int endtime	[DISPLAY_LENGTH];
  byte startnumber	[DISPLAY_LENGTH];
  byte number		[DISPLAY_LENGTH];
  float accel		[DISPLAY_LENGTH];
};

typedef struct tubestmatrix tubetype;
tubetype tubes; // Becuase we only have 1K of ram, the animation array is going be global for now.

void p(prog_uchar *data)
{
  /*
     Print form PROGMEN method - this method uses absolutey no ram, so that's what it's for.

     when using this method, it's better not to break the stings up into shorter bits.
     about 5-6 bytes are added each time you call this function + 1 byte at the end of each string.
    
     taken from http://www.arduino.cc/cgi-bin/yabb2/YaBB.pl?num=1227919972/0
   
     requires "avr/pgmspace.h"
     prog_uchar and pgm_read_byte are from that library.
     see also http://www.arduino.cc/en/Reference/PROGMEM

     define your flash ram strings like this:
	prog_uchar bigString[] PROGMEM = {"123456789abcdefghijklmnopqrstuvwxyz\r\n"};

     You can then print them to the Serial port like this:
        p(bigString);

     Note: "prog_uchar new_line[] PROGMEM  = {"\r\n"};" is a nice idea, but adds 8 bytes to your code
     each time you p(new_line). It's better to add the return & LineFeed to the strings directly.

  */
  
  while(pgm_read_byte(data) != 0x00)
    {
    Serial.print(pgm_read_byte(data++));
    }

}

#ifndef USE_SQUARE_ALGORITHM
  float cube_root(unsigned long a_)
  {
  /*

  Function to return the cube root of a_
  This function is here so we don't need #include <math.h>

  The following optimized C programming language implementation uses Halley's method.
  It demonstrates successively raising an initial approximation by powers of 2,
  until it has a third as many binary digits as the input.
  It works for non-negative integer inputs.

  source: http://en.wikipedia.org/wiki/Cube_root

  */

    unsigned long  x_ = 1;
    float          x, xxx, a;

    if (a_ == 0)
    {
      return 0.0;
    }
    if (a_ == 1)
    {
      return 1.0;
    }

    a = (double) a_;
    do {
      a_ >>= 3;
      x_ <<= 1;
    } 
    while (a_);

    x = (double) x_;
    xxx = x * x * x, x = x * (xxx + a + a) / (xxx + xxx + a);
    /* Accurate to 2 decimal digits: */
    xxx = x * x * x, x = x * (xxx + a + a) / (xxx + xxx + a);
    /* Accurate to 7 decimal digits: */
    xxx = x * x * x, x = x * (xxx + a + a) / (xxx + xxx + a);
    /* Accurate to 15 decimal digits: */
    xxx = x * x * x, x = x * (xxx + a + a) / (xxx + xxx + a);

    return x;
  }

#endif



void randomize_nixie (byte tube)
{
  // This function will randomize just one nixie.
  
  // acceleration
  tubes.accel[tube] = (float)random(MIN_ACCELERATION, MAX_ACCELERATION+1 );

  // start number
  if (random(0,101) < STARTNUM_BIAS)
  {
    tubes.startnumber[tube] = random (0, START_NUM+1);
  }
  else
  {
    tubes.startnumber[tube] = 0;
  }

  //start time
  tubes.starttime[tube] = random (MIN_STARTTIME, MAX_STARTTIME+1) * -1; //start times must always be negative.

  //endtime
#ifdef USE_SQUARE_ALGORITHM
  tubes.endtime[tube] = (in)sqrt(10*tubes.accel[tube]) + random(MIN_ENDTIME, MAX_ENDTIME+1);  //droptime + endtime
#else
  tubes.endtime[tube] = (long)cube_root(10*tubes.accel[tube]/0.01) + random(MIN_ENDTIME, MAX_ENDTIME+1);  //droptime + endtime
#endif	

  // Current time & number
  tubes.time[tube] = tubes.starttime[tube];
  tubes.number[tube] = tubes.startnumber[tube];

}



#ifdef START_WITH_END_NUM
  void set_end_num(byte tube)
  {
    // Forces start number to be END_NUM
    tubes.time[tube] = 1;
    tubes.startnumber[tube] = END_NUM;
    tubes.number[tube] =END_NUM;
  }
#endif



void randomize_everything (byte numberOfTubes)
{
  // This function will randomize all
  
  int count;

  for ( count = 0 ; count < numberOfTubes ; count ++ )
  {
    randomize_nixie (count);
    
#ifdef START_WITH_END_NUM
    set_end_num(count);
#endif

  }

}



void update_tube (byte tube)
{
  // The math to describes how an elements falls is here.

  float d, fraction;
  int dint; // when dint is byte, the output gets all messed up; use int.

  // tube elements only drop when time > 0
  if (tubes.time[tube] < 0)
  {
    dint = tubes.startnumber[tube];
  }
  else
  {
    // Newton: d=0.5at^2
    // cyphropad: d=t^2/a  (or)  d=t^3/a*0.01
    // If you use custom formulas here, remember also put code in randomize_nixie() above for calculation of the end time.
    
#ifdef USE_SQUARE_ALGORITHM
    d = (float)(tubes.time[tube]*tubes.time[tube]) / tubes.accel[tube];
#else
    d = (float)(tubes.time[tube]*tubes.time[tube]*tubes.time[tube]) / tubes.accel[tube]*0.01;
#endif

    if (d < 0) // this may seems redundant, but it's here to catch theoretical float overflow/roll overs.
    {
      dint = tubes.startnumber[tube];
    }
    else if  (d > END_NUM)  // we only have 0-9 elements, so limit d to 10 (10 = can mean a tube that is off)
    {
      dint = END_NUM;
    }
    else
    {
      dint = (int)d;
      // if we don't round the float to byte correctly, the tube elements will appear to fall erratically
      fraction =  d - (float)dint;	
      if (fraction >= 0.5)
      {
        dint =dint +1;
      }
      dint += tubes.startnumber[tube];  // offset to the startnumber and we can start on any element we like
      if (dint > END_NUM )  // catch an overflow if the offset or any other reason causes one.
      {
        dint = END_NUM;
      }
    }
    
  }
  tubes.number[tube] = (byte)dint; // this is the digit that will finally be displayed.
  tubes.time[tube]++; // Update the tube's time.

  if (tubes.time[tube] > tubes.endtime[tube])  //did we reach the end of the nixie element's life?
  {
    if (random(0,101) < CHANGE_BIAS) // each tube only has some chance to change, otherwise it just repeats as it was.
    {
      randomize_nixie(tube);
    }
    else
    {
      tubes.time[tube] = tubes.starttime[tube];
      tubes.number[tube] = dint = tubes.startnumber[tube];
    }

  }

}



void update_all_tubes(byte numberOfTubes)
{

  byte count;
  for ( count = 0 ; count < numberOfTubes ; count++ )
  {
    update_tube(count);
  }

}



char read_Serial_key()
{
  // Function to get the last key pressed on the keyboard (from the serial port) and return it.
  // This function does not stop and wait for a key. If a key is not press '\0' is retuned.
  
  char val = '\0';

  if (Serial.available())
  {
    val = Serial.read();
  }

  return val;

}


void print_run_vals(char key)
{
  // On screen (via serial port) menu.
  
  LED_ON;

  Serial.print(key);
  p(SpeedMenu); // " Speed [-/+] = "
  Serial.print(SPEED, DEC);
  p(MinAccelMenu); // "   MinAccel [9/0] = "
  Serial.print(MIN_ACCELERATION, DEC);
  p(MaxAccelMenu); // "   MaxAccel [(/)] = "
  Serial.println(MAX_ACCELERATION, DEC);

  p(StartTimeMenu); // "  MIN/MAX StartTime [a/s]&[A/S] = "
  Serial.print(MIN_STARTTIME, DEC);
  p(AndMenu); // (" & "
  Serial.println(MAX_STARTTIME, DEC);

  Serial.print("  MIN/MAX EndTime   [z/x]&[Z/X] = ");
  Serial.print(MIN_ENDTIME, DEC);
  p(AndMenu); //  " & "
  Serial.println(MAX_ENDTIME, DEC);

  p(HighestStartMenu); // "  Highest start# [1/2] = "
  Serial.print(START_NUM, DEC);
  p(BiasMenu); // "  #Bias [!/@] = "
  Serial.print(CHANGE_BIAS, DEC);
  p(EndMenu); // "  End# [3/4] = "
  Serial.println(END_NUM, DEC);

  p(ChnageBiasMenu); // "  Change Bias [q/w] = "
  Serial.println(STARTNUM_BIAS, DEC);

  LED_OFF;

}



void read_Serial_keyboard()
{
// Function to get live user input from the serial port and adjust runtime values.
  
#define speedSteps	1;
#define AccelSteps	10;
#define timeSteps	10;
#define numSteps	1;
#define biaSteps	5;

  switch (read_Serial_key())
  {

  case '+':
  case '=':
    SPEED -= speedSteps; //less time = faster runtime
    if (SPEED < 0)
    {
      SPEED = 0;
    }
    print_run_vals('+');
    break;

  case '_':
  case '-':
    SPEED += speedSteps; // more time = slower runtime
    print_run_vals('-');
    break;

  case '9' :
    MIN_ACCELERATION -= AccelSteps;
    if ( MIN_ACCELERATION < 0)
    {
      MIN_ACCELERATION = 0;
    }
    print_run_vals('9');
    break;

  case '0' :
    MIN_ACCELERATION += AccelSteps;
    if (MIN_ACCELERATION > MAX_ACCELERATION)
    {
      MIN_ACCELERATION = MAX_ACCELERATION;
    }
    print_run_vals('0');
    break;

  case '(' :
    MAX_ACCELERATION -= AccelSteps;
    if (MAX_ACCELERATION < MIN_ACCELERATION)
    {
      MAX_ACCELERATION = MIN_ACCELERATION;
    }
    print_run_vals('(');
    break;

  case ')':
    MAX_ACCELERATION += AccelSteps;
    print_run_vals(')');
    break;


  case 'a':
    MIN_STARTTIME -= timeSteps;
    if ( MIN_STARTTIME < 0)
    {
      MIN_STARTTIME = 0;
    }
    print_run_vals('a');
    break;

  case 's':
    MIN_STARTTIME += timeSteps;
    if (MIN_STARTTIME > MAX_STARTTIME)
    {
      MIN_STARTTIME = MAX_STARTTIME;
    }
    print_run_vals('s');
    break;

  case 'A':
    MAX_STARTTIME -= timeSteps;
    if ( MAX_STARTTIME < MIN_STARTTIME)
    {
      MAX_STARTTIME = MIN_STARTTIME;
    }
    print_run_vals('A');
    break;

  case 'S':
    MAX_STARTTIME += timeSteps;
    print_run_vals('S');
    break;

  case 'z':
    MIN_ENDTIME -= timeSteps;
    if ( MIN_ENDTIME < 0)
    {
      MIN_ENDTIME = 0;
    }
    print_run_vals('z');
    break;

  case 'x':
    MIN_ENDTIME += timeSteps;
    if ( MIN_ENDTIME > MAX_ENDTIME)
    {
      MIN_ENDTIME = MAX_ENDTIME;
    }
    print_run_vals('x');
    break;

  case 'Z':
    MAX_ENDTIME -= timeSteps;
    if ( MAX_ENDTIME < MIN_ENDTIME)
    {
      MAX_ENDTIME = MIN_ENDTIME;
    }
    print_run_vals('Z');
    break;

  case 'X':
    MAX_ENDTIME += timeSteps;
    print_run_vals('X');
    break;

  case '1' :
    START_NUM -= numSteps;
    if (START_NUM < 0)
    {
      START_NUM = 0;
    }
    print_run_vals('1');
    break;

  case '2' :
    START_NUM += numSteps;
    if (START_NUM > END_NUM)
    {
      START_NUM = END_NUM;
    }
    print_run_vals('2');
    break;

  case '!' :
    CHANGE_BIAS -= biaSteps;
    if (CHANGE_BIAS < 0)
    {
      CHANGE_BIAS = 0;
    }
    print_run_vals('q');
    break;

  case '@' :
    CHANGE_BIAS += biaSteps;
    if (CHANGE_BIAS > 100)
    {
      CHANGE_BIAS = 100;
    }
    print_run_vals('w');
    break;

  case '3' :
  case '#' :
    END_NUM -= numSteps;
    if (END_NUM < START_NUM)
    {
      END_NUM = START_NUM;
    }
    print_run_vals('q');
    break;

  case '4' :
  case '$' :
    END_NUM += numSteps;
    if (END_NUM > 10)
    {
      END_NUM = 10;
    }
    print_run_vals('w');
    break;

  case 'q':
  case 'Q':
    STARTNUM_BIAS -= biaSteps;
    if (STARTNUM_BIAS < 0)
    {
      STARTNUM_BIAS = 0;
    }
    print_run_vals('Q');
    break;

  case 'w':
  case 'W':
    STARTNUM_BIAS += biaSteps;
    if (STARTNUM_BIAS > 100)
    {
      STARTNUM_BIAS = 100;
    }
    print_run_vals('W');
    break;
  }

}


#ifndef BOOT_WITHOUT_SELF_TEST
void nixie_self_test(int numberOfTubes)
{
/* Short and sweet Function to test all your tubes & spot code errors & hardware faults.

   The code below is between complier directives so that it can optionally not be compiled at all:
   When the nixie_self_test runs, it will use "number of nixies you have" + 2 bytes.
   If you have a lot nixies, you potentially will not have enough RAM to run this test.
   You may also find it anoying and may rather just have it off.
   
*/
   
#define longDelay 555 //everyon loves a timer :)
  byte x, y;
  int nixieDisplay[DISPLAY_LENGTH];

  LED_ON;
  p(SelfTest);
  Serial.print(numberOfTubes, DEC);
  p(Nixie);
  nixie.clear(numberOfTubes);
  LED_OFF;

  for ( y = 9 ; y < 10 ; y-- )
  {
    for (x = 0 ; x < (numberOfTubes) ; x++ )
    {
      nixieDisplay[x] = y; 
      Serial.print(y, DEC);
      /* optional method.
      nixieDisplay[x] = nixieElementOrder[y]; 
      Serial.print(nixieElementOrder[y]);
      */
    } 
    Serial.println();
    nixie.writeArray(nixieDisplay, DISPLAY_LENGTH);
    LED_ON;
    delay(longDelay/2);
    LED_OFF;
    delay(longDelay/2);
  }

  LED_ON;
  nixie.clear(numberOfTubes);
  delay(longDelay*2);
  LED_OFF;

}
#endif



#ifdef OUTPUT_NIXIE_TO_SERIAL
void nixie_writeArray(int* arrayNums, int size)
{
  // Use for testing / de-bugging.

  int count;
  for (count = 0 ; count < (size) ; count++ )
  {
    if ((arrayNums[count] >= 0) && (arrayNums[count] <= 9))
    {
      Serial.print(arrayNums[count], DEC);
    }
    else
    {
      Serial.print(" ");
    }
  }
  Serial.println();
}
#endif



void setup()
{
  Serial.begin(9600);
  
  p(bootmsg1);
  p(bootmsg2);
  Serial.println(VERSION);
  p(bootmsg3);
  p(bootmsg4);
  p(bootmsg5);
  
  
#ifndef USE_SQUARE_ALGORITHM
  p(CubicAlg);
#endif
  Serial.println();

  pinMode(LED, OUTPUT);
#ifndef BOOT_WITHOUT_SELF_TEST
  nixie_self_test(DISPLAY_LENGTH);	// visually test all tubes and elements
#endif
  nixie.clear(DISPLAY_LENGTH);		// Clear the display if you wish

  randomSeed(analogRead(0));            // If you have a clock, point this to the time.
  randomize_everything(DISPLAY_LENGTH); // Set up all the tube values, only need to do this one time. 

}



void loop()
{
  unsigned int count;
  int x[DISPLAY_LENGTH]; // I know, I know, x as a variable name is haram.

  Serial.print(freemem(), DEC);
  p(BytesFree);
  p(Running);
  

  for (EVER)
  {

    //#1 load x[] with all the current tube element positions.
    for ( count = 0 ; count < DISPLAY_LENGTH ; count++ )
    {
      x[count] = nixieElementOrder[tubes.number[count]];
    }
    //#2 send x[] off the the nixie tubes (this is where you will now finally see something)
    nixie.writeArray(x, DISPLAY_LENGTH);
    
#ifdef OUTPUT_NIXIE_TO_SERIAL
    nixie_writeArray(x, DISPLAY_LENGTH);  // Use for testing.
#endif

    //#3 Check the keyboard for run-time inputs
    read_Serial_keyboard();

    //#4 update time and (re-calculte) all nixie tube element positions.
    update_all_tubes(DISPLAY_LENGTH);
    
    //#5 delay
    delay(SPEED);

  }

}
